<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Backend API Tester</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Raleway', sans-serif;
            background: linear-gradient(135deg, #d4a574 0%, #8b6f47 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        header {
            text-align: center;
            color: white;
            margin-bottom: 40px;
        }
        header h1 {
            font-size: 36px;
            margin-bottom: 10px;
        }
        .test-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        .test-card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.2);
            transition: transform 0.3s ease;
        }
        .test-card:hover {
            transform: translateY(-5px);
        }
        .test-card h2 {
            color: #d4a574;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 20px;
        }
        .test-card i {
            font-size: 24px;
        }
        .test-form {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        input, textarea {
            padding: 10px 12px;
            border: 2px solid #e8e8e8;
            border-radius: 6px;
            font-family: inherit;
            font-size: 14px;
            transition: all 0.3s ease;
        }
        input:focus, textarea:focus {
            border-color: #d4a574;
            box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
            outline: none;
        }
        textarea {
            resize: vertical;
            min-height: 80px;
        }
        button {
            background: linear-gradient(135deg, #d4a574 0%, #8b6f47 100%);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
        }
        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(212, 165, 116, 0.3);
        }
        button:active {
            transform: translateY(0);
        }
        .response-box {
            background: #f9f7f4;
            border: 2px solid #e8e8e8;
            border-radius: 6px;
            padding: 12px;
            margin-top: 12px;
            max-height: 200px;
            overflow-y: auto;
            font-size: 12px;
            font-family: monospace;
        }
        .response-box.success {
            border-color: #27ae60;
            background: #d4edda;
            color: #155724;
        }
        .response-box.error {
            border-color: #c0392b;
            background: #f8d7da;
            color: #721c24;
        }
        .status {
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
            margin-top: 8px;
            display: inline-block;
        }
        .status.loading {
            background: #e8f4f8;
            color: #3498db;
        }
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-top: 8px;
        }
        .info-box {
            background: #e8f4f8;
            border-left: 4px solid #3498db;
            padding: 12px;
            border-radius: 4px;
            font-size: 12px;
            color: #333;
            margin-bottom: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1><i class="fas fa-flask"></i> Backend API Tester</h1>
            <p>Test all backend endpoints in real-time</p>
        </header>

        <div class="test-grid">
            <!-- Sign Up Tester -->
            <div class="test-card">
                <h2><i class="fas fa-user-plus"></i> Sign Up</h2>
                <div class="info-box">
                    Create a new user account
                </div>
                <div class="test-form">
                    <input type="text" id="signupName" placeholder="Full Name" value="Test User">
                    <input type="email" id="signupEmail" placeholder="Email" value="test@example.com">
                    <input type="password" id="signupPassword" placeholder="Password" value="test123456">
                    <input type="password" id="signupConfirm" placeholder="Confirm Password" value="test123456">
                    <button onclick="testSignUp()">
                        <i class="fas fa-paper-plane"></i> Test Sign Up
                    </button>
                    <div id="signupResponse" class="response-box"></div>
                </div>
            </div>

            <!-- Sign In Tester -->
            <div class="test-card">
                <h2><i class="fas fa-sign-in-alt"></i> Sign In</h2>
                <div class="info-box">
                    Login with existing account
                </div>
                <div class="test-form">
                    <input type="email" id="signinEmail" placeholder="Email" value="test@example.com">
                    <input type="password" id="signinPassword" placeholder="Password" value="test123456">
                    <div class="checkbox-group">
                        <input type="checkbox" id="rememberMe">
                        <label for="rememberMe">Remember Me</label>
                    </div>
                    <button onclick="testSignIn()">
                        <i class="fas fa-paper-plane"></i> Test Sign In
                    </button>
                    <div id="signinResponse" class="response-box"></div>
                </div>
            </div>

            <!-- Booking Tester -->
            <div class="test-card">
                <h2><i class="fas fa-calendar"></i> Table Booking</h2>
                <div class="info-box">
                    Create a table reservation
                </div>
                <div class="test-form">
                    <input type="text" id="bookingName" placeholder="Name" value="John Doe">
                    <input type="email" id="bookingEmail" placeholder="Email" value="john@example.com">
                    <input type="tel" id="bookingPhone" placeholder="Phone" value="+1 (555) 000-0000">
                    <input type="date" id="bookingDate" value="">
                    <input type="time" id="bookingTime" value="19:00">
                    <input type="number" id="bookingGuests" placeholder="Number of Guests" value="4" min="1" max="100">
                    <button onclick="testBooking()">
                        <i class="fas fa-paper-plane"></i> Test Booking
                    </button>
                    <div id="bookingResponse" class="response-box"></div>
                </div>
            </div>

            <!-- Logout Tester -->
            <div class="test-card">
                <h2><i class="fas fa-sign-out-alt"></i> Logout</h2>
                <div class="info-box">
                    Destroy user session
                </div>
                <div class="test-form">
                    <button onclick="testLogout()">
                        <i class="fas fa-paper-plane"></i> Test Logout
                    </button>
                    <div id="logoutResponse" class="response-box"></div>
                </div>
            </div>

            <!-- Database Status -->
            <div class="test-card">
                <h2><i class="fas fa-database"></i> Database Status</h2>
                <div class="info-box">
                    Check database connection
                </div>
                <div class="test-form">
                    <button onclick="testDatabase()">
                        <i class="fas fa-check-circle"></i> Check Database
                    </button>
                    <div id="dbResponse" class="response-box"></div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="test-card">
                <h2><i class="fas fa-bolt"></i> Quick Actions</h2>
                <div class="info-box">
                    Useful utilities
                </div>
                <div class="test-form">
                    <button onclick="clearLocalStorage()">
                        <i class="fas fa-trash"></i> Clear Storage
                    </button>
                    <button onclick="viewLocalStorage()">
                        <i class="fas fa-eye"></i> View Storage
                    </button>
                    <button onclick="setMinDate()">
                        <i class="fas fa-calendar"></i> Set Tomorrow
                    </button>
                    <div id="actionResponse" class="response-box"></div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Set minimum date to today
        document.getElementById('bookingDate').min = new Date().toISOString().split('T')[0];

        function showResponse(elementId, data, isSuccess = true) {
            const element = document.getElementById(elementId);
            element.className = `response-box ${isSuccess ? 'success' : 'error'}`;
            element.textContent = JSON.stringify(data, null, 2);
        }

        function testSignUp() {
            const data = {
                name: document.getElementById('signupName').value,
                email: document.getElementById('signupEmail').value,
                password: document.getElementById('signupPassword').value,
                confirmPassword: document.getElementById('signupConfirm').value
            };

            fetch('signup.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            })
            .then(r => r.json())
            .then(d => showResponse('signupResponse', d, d.success))
            .catch(e => showResponse('signupResponse', { error: e.message }, false));
        }

        function testSignIn() {
            const data = {
                email: document.getElementById('signinEmail').value,
                password: document.getElementById('signinPassword').value,
                rememberMe: document.getElementById('rememberMe').checked
            };

            fetch('signin.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            })
            .then(r => r.json())
            .then(d => showResponse('signinResponse', d, d.success))
            .catch(e => showResponse('signinResponse', { error: e.message }, false));
        }

        function testBooking() {
            const data = {
                booking_name: document.getElementById('bookingName').value,
                booking_email: document.getElementById('bookingEmail').value,
                booking_phone: document.getElementById('bookingPhone').value,
                booking_date: document.getElementById('bookingDate').value,
                booking_time: document.getElementById('bookingTime').value,
                booking_guests: document.getElementById('bookingGuests').value
            };

            fetch('booking.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            })
            .then(r => r.json())
            .then(d => showResponse('bookingResponse', d, d.success))
            .catch(e => showResponse('bookingResponse', { error: e.message }, false));
        }

        function testLogout() {
            fetch('logout.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' }
            })
            .then(r => r.json())
            .then(d => showResponse('logoutResponse', d, d.success))
            .catch(e => showResponse('logoutResponse', { error: e.message }, false));
        }

        function testDatabase() {
            fetch('config.php', { method: 'POST' })
            .then(r => r.text())
            .then(d => {
                showResponse('dbResponse', { status: 'Database connection OK' }, true);
            })
            .catch(e => showResponse('dbResponse', { error: 'Database connection failed' }, false));
        }

        function clearLocalStorage() {
            localStorage.clear();
            showResponse('actionResponse', { message: 'LocalStorage cleared' }, true);
        }

        function viewLocalStorage() {
            const user = JSON.parse(localStorage.getItem('user') || '{}');
            const booking = JSON.parse(localStorage.getItem('lastBooking') || '{}');
            showResponse('actionResponse', { user, lastBooking: booking }, true);
        }

        function setMinDate() {
            const tomorrow = new Date();
            tomorrow.setDate(tomorrow.getDate() + 1);
            document.getElementById('bookingDate').value = tomorrow.toISOString().split('T')[0];
            showResponse('actionResponse', { message: 'Date set to tomorrow' }, true);
        }
    </script>
</body>
</html>
