<?php
/**
 * Integration Test: Complete Server Login & Dashboard Flow
 * This test simulates the complete user journey
 */

session_start();

include 'config.php';

?>
<!DOCTYPE html>
<html>
<head>
    <title>Server Module Integration Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .test-section {
            margin: 20px 0;
            padding: 15px;
            border-left: 4px solid #2196F3;
            background: #f9f9f9;
        }
        .test-section h3 {
            margin-top: 0;
            color: #2196F3;
        }
        .status {
            padding: 10px;
            border-radius: 4px;
            margin: 10px 0;
            font-family: monospace;
        }
        .status.success {
            background: #c8e6c9;
            color: #2e7d32;
            border: 1px solid #2e7d32;
        }
        .status.error {
            background: #ffcdd2;
            color: #c62828;
            border: 1px solid #c62828;
        }
        .status.info {
            background: #bbdefb;
            color: #1565c0;
            border: 1px solid #1565c0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 10px 0;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background: #f5f5f5;
            font-weight: bold;
        }
        code {
            background: #f5f5f5;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Server Module Integration Test</h1>
        <p>Testing the complete server login and dashboard flow</p>

        <?php
        // Clear any existing session
        $_SESSION = [];
        session_destroy();
        session_start();

        // Test 1: Verify server exists
        ?>
        <div class="test-section">
            <h3>Test 1: Verify Test Server Exists</h3>
            <?php
            $test_employee_id = 'SRV002';
            $test_password = 'Server@123';
            
            $sql = "SELECT id, name, phone, employee_id, status FROM servers WHERE employee_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("s", $test_employee_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $server = $result->fetch_assoc();
                echo "<div class='status success'>✓ Server found: " . $server['name'] . " (ID: " . $server['id'] . ")</div>";
                $server_id = $server['id'];
            } else {
                echo "<div class='status error'>✗ Server not found with employee_id: " . $test_employee_id . "</div>";
                exit;
            }
            ?>
        </div>

        <?php
        // Test 2: Verify password
        ?>
        <div class="test-section">
            <h3>Test 2: Verify Password Hash</h3>
            <?php
            $sql = "SELECT password FROM servers WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $server_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $server_data = $result->fetch_assoc();
            
            if (password_verify($test_password, $server_data['password'])) {
                echo "<div class='status success'>✓ Password verified</div>";
            } else {
                echo "<div class='status error'>✗ Password verification failed</div>";
                exit;
            }
            ?>
        </div>

        <?php
        // Test 3: Simulate session login
        ?>
        <div class="test-section">
            <h3>Test 3: Simulate Login (Set Session)</h3>
            <?php
            $_SESSION['server_id'] = $server_id;
            $_SESSION['server_name'] = $server['name'];
            $_SESSION['server_employee_id'] = $server['employee_id'];
            $_SESSION['server_phone'] = $server['phone'];
            $_SESSION['last_activity'] = time();
            
            if (isset($_SESSION['server_id'])) {
                echo "<div class='status success'>✓ Session created for server ID: " . $_SESSION['server_id'] . "</div>";
            } else {
                echo "<div class='status error'>✗ Failed to create session</div>";
                exit;
            }
            ?>
        </div>

        <?php
        // Test 4: Check assigned tables
        ?>
        <div class="test-section">
            <h3>Test 4: Get Assigned Tables</h3>
            <?php
            $tables_sql = "SELECT 
                            ta.id as assignment_id,
                            ta.table_id,
                            ta.server_id,
                            ta.status as assignment_status,
                            ta.number_of_guests,
                            ta.assigned_at,
                            rt.table_number,
                            rt.capacity,
                            rt.floor,
                            rt.section,
                            rt.status as table_status
                           FROM table_assignments ta
                           INNER JOIN restaurant_tables rt ON ta.table_id = rt.id
                           WHERE ta.server_id = " . intval($server_id) . " 
                           AND ta.status = 'assigned'
                           ORDER BY rt.table_number";
            
            echo "<code>" . htmlspecialchars($tables_sql) . "</code>";
            echo "<br><br>";
            
            $tables_result = $conn->query($tables_sql);
            
            if (!$tables_result) {
                echo "<div class='status error'>✗ Query failed: " . $conn->error . "</div>";
            } else {
                $count = $tables_result->num_rows;
                if ($count > 0) {
                    echo "<div class='status success'>✓ Found " . $count . " assigned table(s)</div>";
                    
                    echo "<table>";
                    echo "<tr><th>Assignment ID</th><th>Table #</th><th>Capacity</th><th>Floor</th><th>Section</th><th>Status</th><th>Guests</th></tr>";
                    
                    while ($table = $tables_result->fetch_assoc()) {
                        echo "<tr>";
                        echo "<td>" . $table['assignment_id'] . "</td>";
                        echo "<td>" . $table['table_number'] . "</td>";
                        echo "<td>" . $table['capacity'] . "</td>";
                        echo "<td>" . $table['floor'] . "</td>";
                        echo "<td>" . $table['section'] . "</td>";
                        echo "<td>" . $table['assignment_status'] . "</td>";
                        echo "<td>" . ($table['number_of_guests'] ?? 'N/A') . "</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                } else {
                    echo "<div class='status error'>✗ No assigned tables found for this server</div>";
                }
            }
            ?>
        </div>

        <?php
        // Test 5: Simulate API response
        ?>
        <div class="test-section">
            <h3>Test 5: Simulate API Response (What Dashboard Would Receive)</h3>
            <?php
            // Re-fetch the tables for formatting
            $tables_result = $conn->query($tables_sql);
            $assigned_tables = [];
            
            if ($tables_result && $tables_result->num_rows > 0) {
                while ($table = $tables_result->fetch_assoc()) {
                    $assigned_tables[] = [
                        'id' => $table['table_id'],  // Use table_id as the id for frontend operations
                        'table_number' => $table['table_number'],
                        'capacity' => $table['capacity'],
                        'floor' => $table['floor'],
                        'section' => $table['section'],
                        'status' => $table['assignment_status'],
                        'number_of_guests' => $table['number_of_guests'],
                        'assigned_at' => $table['assigned_at']
                    ];
                }
            }
            
            $response_data = [
                'id' => $server['id'],
                'name' => $server['name'],
                'phone' => $server['phone'],
                'employee_id' => $server['employee_id'],
                'status' => $server['status'],
                'is_online' => (bool) ($server['is_online'] ?? false),
                'assigned_tables' => $assigned_tables,
                'assigned_tables_count' => count($assigned_tables)
            ];
            
            echo "<pre><code>" . json_encode($response_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . "</code></pre>";
            ?>
        </div>

        <?php
        // Test 6: Verify frontend will receive correct data
        ?>
        <div class="test-section">
            <h3>Test 6: Frontend Integration</h3>
            <?php
            if (count($assigned_tables) > 0) {
                echo "<div class='status success'>✓ Dashboard will receive " . count($assigned_tables) . " table(s) for rendering</div>";
                echo "<p>Each table card will display:</p>";
                echo "<ul>";
                foreach ($assigned_tables as $table) {
                    echo "<li>Table " . $table['table_number'] . " (ID: " . $table['id'] . ") - Floor " . $table['floor'] . ", Section " . $table['section'] . ", Capacity: " . $table['capacity'] . "</li>";
                }
                echo "</ul>";
            } else {
                echo "<div class='status error'>✗ No tables to display on dashboard</div>";
            }
            ?>
        </div>

        <div class="test-section" style="border-left-color: #4CAF50;">
            <h3 style="color: #4CAF50;">✓ Test Complete</h3>
            <p>The server module is ready for testing. Use the following credentials:</p>
            <table>
                <tr><th>Employee ID</th><th>Password</th></tr>
                <tr><td><?php echo $test_employee_id; ?></td><td><?php echo $test_password; ?></td></tr>
            </table>
            <p><a href="../server/dashboard.php" style="color: #2196F3; text-decoration: none;">→ Go to Server Dashboard</a> (you may need to login first)</p>
        </div>
    </div>
</body>
</html>
