<?php
/**
 * Kitchen Staff Test Data Setup
 * Creates sample kitchen staff accounts for testing
 * Access: /rest_table/backend/setup_kitchen_staff.php?admin_key=granny_admin_2026
 */

header('Content-Type: application/json');
require_once 'config.php';

// Simple admin key verification (change in production)
$admin_key = $_GET['admin_key'] ?? '';
$expected_key = 'granny_admin_2026'; // Change this to a secure key

if ($admin_key !== $expected_key) {
    http_response_code(401);
    sendResponse(false, 'Unauthorized: Invalid admin key', null);
}

try {
    $test_staff = [
        [
            'name' => 'Chef Ramesh Kumar',
            'email' => 'ramesh@granny.local',
            'employee_id' => 'CHEF001',
            'position' => 'Chef',
            'role' => 'Chef',
            'password' => 'Chef123!@#',
            'status' => 'active'
        ],
        [
            'name' => 'Sous Chef Priya Singh',
            'email' => 'priya@granny.local',
            'employee_id' => 'SOUS001',
            'position' => 'Sous Chef',
            'role' => 'Chef',
            'password' => 'Sous123!@#',
            'status' => 'active'
        ],
        [
            'name' => 'Line Cook Arjun',
            'email' => 'arjun@granny.local',
            'employee_id' => 'COOK001',
            'position' => 'Line Cook',
            'role' => 'Staff',
            'password' => 'Cook123!@#',
            'status' => 'active'
        ],
        [
            'name' => 'Kitchen Assistant Sunita',
            'email' => 'sunita@granny.local',
            'employee_id' => 'ASST001',
            'position' => 'Kitchen Assistant',
            'role' => 'Staff',
            'password' => 'Asst123!@#',
            'status' => 'active'
        ],
        [
            'name' => 'Pastry Chef Vikram',
            'email' => 'vikram@granny.local',
            'employee_id' => 'PASTRY001',
            'position' => 'Pastry Chef',
            'role' => 'Chef',
            'password' => 'Pastry123!@#',
            'status' => 'active'
        ]
    ];
    
    $created = [];
    $errors = [];
    
    foreach ($test_staff as $staff) {
        // Check if already exists
        $check = $conn->prepare("SELECT id FROM chefs WHERE email = ? OR employee_id = ?");
        $check->bind_param('ss', $staff['email'], $staff['employee_id']);
        $check->execute();
        
        if ($check->get_result()->num_rows > 0) {
            $errors[] = $staff['name'] . ' already exists (Email: ' . $staff['email'] . ')';
            continue;
        }
        
        // Create staff
        $hashed_password = password_hash($staff['password'], PASSWORD_BCRYPT);
        $insert = $conn->prepare(
            "INSERT INTO chefs (name, email, employee_id, position, role, password, status, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, NOW())"
        );
        
        $insert->bind_param(
            'sssssss',
            $staff['name'],
            $staff['email'],
            $staff['employee_id'],
            $staff['position'],
            $staff['role'],
            $hashed_password,
            $staff['status']
        );
        
        if ($insert->execute()) {
            $created[] = [
                'name' => $staff['name'],
                'email' => $staff['email'],
                'employee_id' => $staff['employee_id'],
                'password' => $staff['password'], // Only for setup, never log in production
                'message' => 'Created successfully'
            ];
        } else {
            $errors[] = $staff['name'] . ' - Error: ' . $insert->error;
        }
        
        $insert->close();
        $check->close();
    }
    
    sendResponse(true, 'Test data setup complete', [
        'created' => count($created),
        'staff' => $created,
        'errors' => $errors,
        'total_attempted' => count($test_staff),
        'message' => 'Kitchen staff accounts have been created. Use the email/password combinations shown above to test login at /kitchen/login.php'
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    sendResponse(false, 'Setup failed: ' . $e->getMessage(), null);
}

$conn->close();
?>
