<?php
/**
 * Print KOT Ticket
 * Returns HTML formatted KOT for printing
 * Phase 2: KOT Generation System
 */

header('Content-Type: text/html; charset=utf-8');
require_once 'config.php';

$kot_id = isset($_GET['kot_id']) ? (int)$_GET['kot_id'] : null;
$order_id = isset($_GET['order_id']) ? (int)$_GET['order_id'] : null;

if (!$kot_id && !$order_id) {
    die('Missing kot_id or order_id');
}

// Get KOT details via API logic
if ($kot_id) {
    $kot_sql = "SELECT * FROM kot_tickets WHERE id = ?";
    $stmt = $conn->prepare($kot_sql);
    $stmt->bind_param('i', $kot_id);
} else {
    $kot_sql = "SELECT * FROM kot_tickets WHERE order_id = ?";
    $stmt = $conn->prepare($kot_sql);
    $stmt->bind_param('i', $order_id);
}

$stmt->execute();
$kot_result = $stmt->get_result();

if ($kot_result->num_rows === 0) {
    die('KOT not found');
}

$kot = $kot_result->fetch_assoc();
$stmt->close();
$order_id = $kot['order_id'];

// Get order details
$order_sql = "
    SELECT 
        o.id,
        o.order_number,
        o.table_id,
        o.total_amount,
        o.status,
        o.special_requests,
        rt.table_number,
        rt.section,
        rt.floor_level,
        s.name as server_name
    FROM dine_in_orders o
    LEFT JOIN restaurant_tables rt ON o.table_id = rt.id
    LEFT JOIN servers s ON o.server_id = s.id
    WHERE o.id = ?
";

$stmt = $conn->prepare($order_sql);
$stmt->bind_param('i', $order_id);
$stmt->execute();
$order = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Get order items
$items_sql = "
    SELECT 
        doi.id,
        doi.item_name,
        doi.quantity,
        doi.special_instructions,
        doi.status
    FROM dine_in_order_items doi
    WHERE doi.dine_in_order_id = ?
    ORDER BY doi.created_at ASC
";

$stmt = $conn->prepare($items_sql);
$stmt->bind_param('i', $order_id);
$stmt->execute();
$items_result = $stmt->get_result();

$items = [];
while ($item = $items_result->fetch_assoc()) {
    $items[] = $item;
}
$stmt->close();

$priority_color = [
    'low' => '#4caf50',
    'normal' => '#2196f3',
    'high' => '#ff9800',
    'urgent' => '#f44336'
];

$priority_label = [
    'low' => 'LOW',
    'normal' => 'NORMAL',
    'high' => 'HIGH',
    'urgent' => '🔴 URGENT'
];

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>KOT - Kitchen Order Ticket #<?php echo htmlspecialchars($kot['ticket_number']); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Courier New', monospace;
            background: white;
            padding: 10px;
        }
        
        .kot-container {
            max-width: 400px;
            margin: 0 auto;
            border: 3px solid #333;
            padding: 20px;
            background: white;
        }
        
        .kot-header {
            text-align: center;
            margin-bottom: 15px;
            border-bottom: 2px solid #333;
            padding-bottom: 10px;
        }
        
        .restaurant-name {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .kot-title {
            font-size: 16px;
            font-weight: bold;
            margin: 10px 0;
        }
        
        .ticket-number {
            font-size: 14px;
            font-weight: bold;
            background: #f0f0f0;
            padding: 5px;
            margin: 5px 0;
        }
        
        .priority-badge {
            display: inline-block;
            padding: 8px 12px;
            font-weight: bold;
            color: white;
            border-radius: 3px;
            margin: 5px 0;
            font-size: 14px;
        }
        
        .order-details {
            margin: 15px 0;
            border-bottom: 1px dashed #333;
            padding-bottom: 10px;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            margin: 5px 0;
            font-size: 13px;
        }
        
        .label {
            font-weight: bold;
        }
        
        .table-section {
            margin: 15px 0;
            border-bottom: 2px solid #333;
            padding-bottom: 10px;
        }
        
        .table-number {
            font-size: 20px;
            font-weight: bold;
            text-align: center;
            background: #fffacd;
            padding: 10px;
            margin-bottom: 10px;
        }
        
        .items-list {
            margin: 15px 0;
        }
        
        .item-row {
            display: flex;
            justify-content: space-between;
            margin: 8px 0;
            padding: 8px;
            background: #f9f9f9;
            border-left: 4px solid #d4a574;
        }
        
        .item-name {
            font-weight: bold;
            flex: 1;
        }
        
        .item-qty {
            font-weight: bold;
            min-width: 30px;
            text-align: center;
        }
        
        .item-special {
            width: 100%;
            margin-top: 5px;
            font-size: 11px;
            color: #666;
            background: #fff9e6;
            padding: 5px;
            border-left: 2px solid #ff9800;
            font-style: italic;
        }
        
        .item-status {
            display: inline-block;
            font-size: 10px;
            padding: 2px 5px;
            border-radius: 3px;
            margin-top: 3px;
            font-weight: bold;
        }
        
        .status-pending {
            background: #ffebee;
            color: #c62828;
        }
        
        .status-preparing {
            background: #fff3e0;
            color: #e65100;
        }
        
        .status-ready {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        .special-notes {
            margin: 15px 0;
            padding: 10px;
            background: #fff9e6;
            border: 2px solid #ff9800;
            border-radius: 3px;
            font-size: 12px;
        }
        
        .special-notes-title {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .footer {
            text-align: center;
            margin-top: 15px;
            padding-top: 10px;
            border-top: 2px solid #333;
            font-size: 11px;
        }
        
        .timestamp {
            margin: 5px 0;
            font-size: 12px;
        }
        
        .print-instructions {
            text-align: center;
            margin: 20px 0;
            font-size: 12px;
            color: #666;
        }
        
        @media print {
            body {
                padding: 0;
            }
            
            .print-instructions {
                display: none;
            }
            
            .kot-container {
                max-width: 100%;
                border: 1px solid #000;
                margin: 0;
                padding: 10px;
                page-break-after: always;
            }
        }
    </style>
</head>
<body>
    <div class="print-instructions">
        <button onclick="window.print()">🖨️ Print KOT</button>
        <button onclick="window.close()">Close</button>
    </div>
    
    <div class="kot-container">
        <!-- Header -->
        <div class="kot-header">
            <div class="restaurant-name">🍽️ GRANNY RESTAURANT</div>
            <div class="kot-title">KITCHEN ORDER TICKET</div>
        </div>
        
        <!-- Ticket Info -->
        <div class="ticket-number">#<?php echo htmlspecialchars($kot['ticket_number']); ?></div>
        
        <div class="priority-badge" style="background-color: <?php echo $priority_color[$kot['priority']] ?? '#2196f3'; ?>">
            <?php echo $priority_label[$kot['priority']] ?? 'NORMAL'; ?>
        </div>
        
        <!-- Order Details -->
        <div class="order-details">
            <div class="detail-row">
                <span class="label">ORDER #:</span>
                <span><?php echo htmlspecialchars($order['order_number']); ?></span>
            </div>
            <div class="detail-row">
                <span class="label">SERVER:</span>
                <span><?php echo htmlspecialchars($order['server_name'] ?? 'N/A'); ?></span>
            </div>
            <div class="detail-row">
                <span class="label">TIME:</span>
                <span><?php echo date('H:i:s', strtotime($kot['generated_at'])); ?></span>
            </div>
        </div>
        
        <!-- Table Info -->
        <div class="table-section">
            <div class="table-number">
                TABLE <?php echo htmlspecialchars($order['table_number']); ?>
            </div>
            <div class="detail-row">
                <span class="label">SECTION:</span>
                <span><?php echo htmlspecialchars($order['section'] ?? 'N/A'); ?></span>
            </div>
            <div class="detail-row">
                <span class="label">FLOOR:</span>
                <span><?php echo htmlspecialchars($order['floor_level'] ?? 'N/A'); ?></span>
            </div>
        </div>
        
        <!-- Items -->
        <div class="items-list">
            <div style="text-align: center; font-weight: bold; margin-bottom: 10px; border-bottom: 1px solid #333; padding-bottom: 5px;">
                ITEMS TO PREPARE
            </div>
            
            <?php foreach ($items as $item): ?>
                <div class="item-row">
                    <div class="item-name"><?php echo htmlspecialchars($item['item_name']); ?></div>
                    <div class="item-qty">×<?php echo $item['quantity']; ?></div>
                </div>
                
                <?php if (!empty($item['special_instructions'])): ?>
                    <div class="item-special">
                        📝 <?php echo htmlspecialchars($item['special_instructions']); ?>
                    </div>
                <?php endif; ?>
                
                <div style="margin: 3px 0;">
                    <span class="item-status status-<?php echo $item['status']; ?>">
                        <?php echo strtoupper($item['status']); ?>
                    </span>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Special Notes -->
        <?php if (!empty($order['special_requests'])): ?>
            <div class="special-notes">
                <div class="special-notes-title">⚠️ SPECIAL NOTES:</div>
                <div><?php echo htmlspecialchars($order['special_requests']); ?></div>
            </div>
        <?php endif; ?>
        
        <!-- Footer -->
        <div class="footer">
            <div class="timestamp">Generated: <?php echo date('d/m/Y H:i:s', strtotime($kot['generated_at'])); ?></div>
            <div style="margin-top: 10px; font-weight: bold;">
                ✓ Please confirm receipt and mark as acknowledged
            </div>
        </div>
    </div>
</body>
</html>

<?php $conn->close(); ?>
