<?php
/**
 * Kitchen Get Pending Orders API
 * Fetches all orders that have been sent to the kitchen and are pending/preparing
 */

session_start();
header('Content-Type: application/json');
require_once 'config.php';

// This is a public API for kitchen staff (can be accessed with kitchen_id, or admin authentication)
// For now, we'll allow basic authentication

try {
    // Get all orders that are confirmed, preparing, or ready (not yet picked up)
    // Only show items with 'pending' or 'preparing' status (not 'ready' from previous kitchen cycles)
    // This prevents showing items that were already prepared and need to be picked up separately
    $sql = "
        SELECT 
            o.id,
            o.order_number,
            o.table_id,
            rt.table_number,
            rt.section,
            rt.floor_level,
            o.server_id,
            s.name as server_name,
            o.total_amount,
            o.status,
            o.special_requests,
            o.created_at,
            (SELECT COUNT(*) FROM dine_in_order_items WHERE dine_in_order_id = o.id AND status IN ('pending', 'preparing')) as item_count,
            (SELECT GROUP_CONCAT(CONCAT(item_name, ' (x', quantity, ')') SEPARATOR ', ') FROM dine_in_order_items WHERE dine_in_order_id = o.id AND status IN ('pending', 'preparing')) as items_summary
        FROM dine_in_orders o
        LEFT JOIN restaurant_tables rt ON o.table_id = rt.id
        LEFT JOIN servers s ON o.server_id = s.id
        WHERE o.status IN ('confirmed', 'preparing', 'ready')
            AND EXISTS (
                SELECT 1 FROM dine_in_order_items WHERE dine_in_order_id = o.id AND status IN ('pending', 'preparing')
            )
        ORDER BY o.created_at ASC
    ";
    
    $result = $conn->query($sql);
    
    if (!$result) {
        throw new Exception('Database query failed: ' . $conn->error);
    }
    
    $orders = [];
    while ($row = $result->fetch_assoc()) {
        $orders[] = [
            'id' => (int)$row['id'],
            'order_number' => $row['order_number'],
            'table_id' => (int)$row['table_id'],
            'table_number' => $row['table_number'],
            'section' => $row['section'],
            'floor' => $row['floor_level'],
            'server_id' => (int)$row['server_id'],
            'server_name' => $row['server_name'],
            'total_amount' => (float)$row['total_amount'],
            'status' => $row['status'],
            'special_requests' => $row['special_requests'],
            'item_count' => (int)$row['item_count'],
            'items_summary' => $row['items_summary'],
            'created_at' => $row['created_at']
        ];
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Orders fetched successfully',
        'data' => $orders,
        'count' => count($orders)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
